/*
 * @Author: zhoufei
 * @Date: 2022-04-06 10:18:28
 * @LastEditors: zhoufei
 * @LastEditTime: 2022-05-13 10:10:14
 * @FilePath: /utils/auto_gc_map.go
 * @Description:
 *
 * Copyright (c) 2022 by zhoufei, All Rights Reserved.
 */
package utils

import (
	"sync"
	"time"
)

const (
	// NeverDie means value.alive() returns true forever.
	NeverDie = 0
)

type value struct {
	data  interface{}
	ttl   int64 //time.Second
	ctime time.Time
}

func newValue(data interface{}, ttl int64) *value {
	return &value{
		data:  data,
		ttl:   ttl,
		ctime: time.Now(),
	}
}

func (v *value) alive() bool {
	return v.ttl == NeverDie || time.Now().Before(v.ctime.Add(time.Duration(v.ttl)*time.Second))
}

type AutoGcMap struct {
	data map[string]*value
	lock *sync.RWMutex
}

func NewAutoGcMap() *AutoGcMap {
	return &AutoGcMap{
		data: make(map[string]*value),
		lock: &sync.RWMutex{},
	}
}

func (my *AutoGcMap) Set(key string, data interface{}, ttl int64) {
	my.lock.Lock()
	defer my.lock.Unlock()
	my.data[key] = newValue(data, ttl)
}

func (my *AutoGcMap) Get(key string) (interface{}, bool) {
	my.lock.Lock()
	defer my.lock.Unlock()
	if value, ok := my.data[key]; ok && value.alive() {
		return value.data, true
	}
	return nil, false
}

func (my *AutoGcMap) Gc() {
	my.lock.Lock()
	defer my.lock.Unlock()
	for key, value := range my.data {
		if !value.alive() {
			delete(my.data, key)
		}
	}

}

func (my *AutoGcMap) AutoGc(duration time.Duration) {
	go func() {
		ticker := time.NewTicker(duration)
		for {
			<-ticker.C
			my.Gc()
		}
	}()
}
