package utils

import (
	"os"

	"github.com/natefinch/lumberjack"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

const (
	// DebugLevel logs are typically voluminous, and are usually disabled in
	// production.
	Debug int8 = iota - 1
	// InfoLevel is the default logging priority.
	Info
	// WarnLevel logs are more important than Info, but don't need individual
	// human review.
	Warn
	// ErrorLevel logs are high-priority. If an application is running smoothly,
	// it shouldn't generate any error-level logs.
	Error
	// DPanicLevel logs are particularly important errors. In development the
	// logger panics after writing the message.
	DPanic
	// PanicLevel logs a message, then panics.
	Panic
	// FatalLevel logs a message, then calls os.Exit(1).
	Fatal
)

type loggerConfig struct {
	Level      int8
	FilePath   string
	MaxSize    int
	MaxBackups int
	MaxAge     int
	Compress   bool
	Console    bool
}

type loggerOption func(m *loggerConfig)

func WithLevel(level int8) loggerOption {
	return func(m *loggerConfig) {
		m.Level = level
	}
}

func WithLogFilePath(filePath string) loggerOption {
	return func(m *loggerConfig) {
		m.FilePath = filePath
	}
}

func WithMaxSize(maxSize int) loggerOption {
	return func(m *loggerConfig) {
		m.MaxSize = maxSize
	}
}

func WithMaxBackups(maxBack int) loggerOption {
	return func(m *loggerConfig) {
		m.MaxBackups = maxBack
	}
}

func WithMaxAge(maxAge int) loggerOption {
	return func(m *loggerConfig) {
		m.MaxAge = maxAge
	}
}

func WithCompress(compress bool) loggerOption {
	return func(m *loggerConfig) {
		m.Compress = compress
	}
}

func WithConsole(isConsole bool) loggerOption {
	return func(m *loggerConfig) {
		m.Console = isConsole
	}
}

var lf = loggerConfig{
	Level:      1,
	FilePath:   "log/log.log",
	MaxSize:    100,
	MaxAge:     7,
	MaxBackups: 7,
	Compress:   true,
}

var L *zap.SugaredLogger

func InitLogger(opt ...loggerOption) {

	for _, config := range opt {
		config(&lf)
	}

	writeSync := getLogWriter()
	encoder := getEncoder()
	var core zapcore.Core
	if lf.Console {
		consoleDebugging := zapcore.Lock(os.Stdout)
		core = zapcore.NewTee(
			zapcore.NewCore(encoder, writeSync, zapcore.Level(lf.Level)),
			zapcore.NewCore(zapcore.NewConsoleEncoder(zap.NewDevelopmentEncoderConfig()), consoleDebugging, zapcore.Level(lf.Level)),
		)
	} else {
		core = zapcore.NewTee(
			zapcore.NewCore(encoder, writeSync, zapcore.Level(lf.Level)),
		)
	}

	logger := zap.New(core, zap.AddCaller())
	L = logger.Sugar()
}

func getEncoder() zapcore.Encoder {
	encoderConfig := zap.NewProductionEncoderConfig()
	encoderConfig.EncodeTime = zapcore.ISO8601TimeEncoder
	encoderConfig.EncodeLevel = zapcore.CapitalLevelEncoder
	return zapcore.NewConsoleEncoder(encoderConfig)
}

func getLogWriter() zapcore.WriteSyncer {
	lumberJackLogger := &lumberjack.Logger{
		Filename:   lf.FilePath,   // 日志文件路径
		MaxSize:    lf.MaxSize,    // 每个日志文件保存的最大尺寸 单位：M
		MaxBackups: lf.MaxBackups, // 日志文件最多保存多少个备份
		MaxAge:     lf.MaxAge,     // 文件最多保存多少天
		Compress:   lf.Compress,   // 是否压缩
	}

	return zapcore.AddSync(lumberJackLogger)
}
