/*
 * @Author: zhoufei
 * @Date: 2022-05-19 15:41:06
 * @LastEditors: zhoufei
 * @LastEditTime: 2022-06-17 11:01:56
 * @FilePath: /sdc/Users/software/Documents/go/utils/map_g.go
 * @Description: 泛型map
 *
 * Copyright (c) 2022 by zhoufei, All Rights Reserved.
 */
package utils

import (
	"sync"
	"time"
)

const (
	// NeverDie means value.alive() returns true forever.
	NeverDieG = 0
)

type valueG[T any] struct {
	data  T
	ttl   int64
	ctime time.Time
}

func newValueG[T any](data T, ttl int64) *valueG[T] {
	return &valueG[T]{data: data, ttl: ttl, ctime: time.Now()}
}

func (v *valueG[T]) alive() bool {
	return v.ttl == NeverDieG || time.Now().Before(v.ctime.Add(time.Duration(v.ttl)*time.Second))
}

type Map[K comparable, T any] struct {
	data map[K]*valueG[T]
	lock *sync.RWMutex
}

func NewMap[K comparable, T any]() *Map[K, T] {
	return &Map[K, T]{data: make(map[K]*valueG[T]), lock: &sync.RWMutex{}}
}

func (m *Map[K, T]) Set(key K, data T, ttl int64) {
	m.lock.Lock()
	defer m.lock.Unlock()
	m.data[key] = newValueG(data, ttl)
}

func (m *Map[K, T]) Get(key K) (T, bool) {
	var d T
	m.lock.Lock()
	defer m.lock.Unlock()
	if value, ok := m.data[key]; ok {
		return value.data, true
	}

	return d, false
}

func (m *Map[K, T]) Len() int {
	m.lock.Lock()
	defer m.lock.Unlock()
	return len(m.data)
}

func (m *Map[K, T]) Keys() []K {
	m.lock.Lock()
	defer m.lock.Unlock()
	keys := make([]K, 0, len(m.data))
	for k := range m.data {
		keys = append(keys, k)
	}
	return keys
}

func (m *Map[K, T]) Values() []T {
	m.lock.Lock()
	defer m.lock.Unlock()
	values := make([]T, 0, len(m.data))
	for _, v := range m.data {
		values = append(values, v.data)
	}
	return values
}

func (m *Map[K, T]) ValueMap() (rm map[K]T) {
	m.lock.Lock()
	defer m.lock.Unlock()
	rm = make(map[K]T)
	for k, v := range m.data {
		rm[k] = v.data
	}
	return
}

func (m *Map[K, T]) CloneTo(another *Map[K, T]) {
	for k := range m.data {
		another.data[k] = m.data[k]
	}
}

func (m *Map[K, T]) Del(k K) {
	m.lock.Lock()
	defer m.lock.Unlock()
	delete(m.data, k)
}

func (m *Map[K, T]) Gc() {
	m.lock.Lock()
	defer m.lock.Unlock()
	for k, v := range m.data {
		if !v.alive() {
			delete(m.data, k)
		}
	}
}

func (m *Map[K, T]) AutoGc(duration time.Duration) {
	go func() {
		ticker := time.NewTicker(duration)
		for {
			<-ticker.C
			m.Gc()
		}
	}()
}
