package utils

import (
	"context"
	"fmt"
	"github.com/qiniu/qmgo"
	"net/url"
)

type MongoConfig struct {
	Username string
	Password string
	Host string
	DB string
	Collection string
	MaxPoolSize uint64
	MinPoolSize uint64

}

type Mongo struct{
	Cli *qmgo.QmgoClient  //必须初始化DB和Collection
	Client *qmgo.Client	// 不用初始化DB和Collection
}

type Option func(m *MongoConfig)


func WithDB(db string)Option{
	return func(m *MongoConfig) {
		m.DB = db
	}
}

func WithCollection(collection string)Option{
	return func(m *MongoConfig) {
		m.Collection = collection
	}
}

func WithMaxPoolSize(poolSize uint64)Option{
	return func(m *MongoConfig){
		m.MaxPoolSize = poolSize
	}
}

func WithMinPoolSize(poolSize uint64)Option{
	return func(m *MongoConfig){
		m.MinPoolSize = poolSize
	}
}

func NewMongo(host string,username string,password string,options ...Option) (m *Mongo, err error) {

	mfg := MongoConfig{
		Username:url.QueryEscape(username),
		Password:url.QueryEscape(password),
		Host:host,
	}

	for _, option := range options {
		option(&mfg)
	}

	cfg := qmgo.Config{
		Uri:fmt.Sprintf("mongodb://%s:%s@%s", mfg.Username, mfg.Password, mfg.Host),
	}
	var maxPoolSize uint64 = 100
	var minPoolSize uint64 = 0
	if mfg.MinPoolSize != 0 {
		cfg.MaxPoolSize = &mfg.MaxPoolSize
	}else{

		cfg.MaxPoolSize = &maxPoolSize
	}

	if mfg.MinPoolSize != 0{
		cfg.MinPoolSize = &mfg.MinPoolSize
	}else{

		cfg.MinPoolSize = &minPoolSize
	}

	var isCollection bool

	if mfg.DB != ""{
		cfg.Database = mfg.DB
	}

	if mfg.Collection != ""{
		cfg.Coll = mfg.Collection
		isCollection = true
	}

	ctx := context.Background()
	m = &Mongo{}
	if isCollection{
		m.Cli,err = qmgo.Open(ctx, &cfg)
	}else{
		m.Client,err = qmgo.NewClient(ctx,&cfg)
	}

	return
}

func (m *Mongo)Close() {
	if m.Client != nil {
		m.Client.Close(context.Background())
	}

	if m.Cli != nil {
		m.Cli.Close(context.Background())
	}

}

