/*
 * @Author: zhoufei
 * @Date: 2022-04-22 14:58:25
 * @LastEditors: zhoufei
 * @LastEditTime: 2022-05-31 16:53:30
 * @FilePath: /utils/queue/rocketmq.go
 * @Description:rocketmq 工具类
 * 注意：rocketmq-client-go/v2 在1.18下会闪退，需要手动更改json-iterator/go 到 v1.1.12 (2022-4-25)
 * Copyright (c) 2022 by zhoufei, All Rights Reserved.
 */
package queue

import (
	"context"
	"errors"
	"fmt"

	"github.com/apache/rocketmq-client-go/v2"
	"github.com/apache/rocketmq-client-go/v2/consumer"
	"github.com/apache/rocketmq-client-go/v2/primitive"
	"github.com/apache/rocketmq-client-go/v2/producer"
	"github.com/apache/rocketmq-client-go/v2/rlog"
)

type RocketMq struct {
	endPoints []string
	producer  rocketmq.Producer
	consumer  rocketmq.PushConsumer
}

type rocketOptions struct {
	AccessKey string
	SecretKey string
}

type RocketOption func(*rocketOptions)

func WithRocketAcl(access string, secret string) RocketOption {
	return func(o *rocketOptions) {
		if access == "" || secret == "" {
			return
		}
		o.AccessKey = access
		o.SecretKey = secret
	}
}

func init() {
	rlog.SetLogLevel("error")
}

/**
 * @description: 发送消息
 * @param {string} topic
 * @param {[]byte} msg
 * @return {*}
 */
func (r *RocketMq) SendMsg(topic string, msg []byte, opts ...MsgOption) (qMsg *QueueMsg, err error) {
	if r.producer == nil {
		return nil, errors.New("no producer available")
	}

	rMsg := primitive.NewMessage(topic, msg)

	msgOptions := &msgOptions{}
	for _, opt := range opts {
		opt(msgOptions)
	}

	var tag string

	if msgOptions.Tag != "" {
		rMsg.WithTag(msgOptions.Tag)
		tag = msgOptions.Tag
	}

	if msgOptions.UserProperties != nil {
		rMsg.WithProperties(msgOptions.UserProperties)
	}

	resp, err := r.producer.SendSync(context.Background(), rMsg)

	if err != nil {
		return
	}
	if resp.Status != primitive.SendOK {
		return nil, fmt.Errorf("error sending message,status:%v", resp.Status)
	}
	qmsg := &QueueMsg{
		Body:  []byte(msg),
		Topic: topic,
		Info:  map[string]interface{}{"msgId": resp.MsgID},
		Tag:   tag,
	}
	return qmsg, err
}

/**
 * @description: 发送字符串消息
 * @param {string} topic
 * @param {string} msg
 * @return {*}
 */
func (r *RocketMq) SendStringMsg(topic string, msg string, opts ...MsgOption) (qMsg *QueueMsg, err error) {
	return r.SendMsg(topic, []byte(msg), opts...)
}

/**
 * @description: 注册消费者回调
 * @param {*QueueMsg} qMsg
 * @return {*}
 */
func (r *RocketMq) RegisterConsumer(topic string, recevieFunc func(qMsg *QueueMsg), opts ...MsgOption) (err error) {
	if r.consumer == nil {
		return errors.New("no consumer available")
	}

	msgOptions := &msgOptions{}
	for _, opt := range opts {
		opt(msgOptions)
	}
	var tag string
	selector := consumer.MessageSelector{}
	if msgOptions.Tag != "" {
		selector.Type = consumer.TAG
		selector.Expression = msgOptions.Tag
		tag = msgOptions.Tag
	} else if msgOptions.Sql92 != "" {
		selector.Type = consumer.SQL92
		selector.Expression = msgOptions.Sql92
	}

	err = r.consumer.Subscribe(topic, selector, func(_ context.Context, msgs ...*primitive.MessageExt) (consumer.ConsumeResult, error) {
		for _, msg := range msgs {
			go recevieFunc(&QueueMsg{
				Body:  msg.Body,
				Topic: topic,
				Info:  map[string]interface{}{"msgId": msg.MsgId},
				Tag:   tag,
			})
		}
		return consumer.ConsumeSuccess, nil
	})

	if err != nil {
		return err
	}

	err = r.consumer.Start()
	if err != nil {
		err = r.consumer.Unsubscribe(topic)
		return err
	}
	return nil
}

func (r *RocketMq) ShutDown() error {
	var err error
	if r.producer != nil {
		err = r.producer.Shutdown()
	}
	if r.consumer != nil {
		err = r.consumer.Shutdown()
	}
	return err
}

/**
 * @description: 创建rocketmq生产者
 * @param {[]string} endPoints nameserver
 * @param {string} groupName
 * @param {int} retry
 * @return {*}
 */
func NewRocketMqProducer(endPoints []string, groupName string, retry int, options ...RocketOption) (p Producer, err error) {

	if retry <= 0 {
		retry = 0
	}

	opts := []producer.Option{producer.WithGroupName(groupName),
		producer.WithNsResolver(primitive.NewPassthroughResolver(endPoints)),
		producer.WithRetry(retry)}

	ropts := &rocketOptions{}
	for _, v := range options {
		v(ropts)
	}

	if ropts.AccessKey != "" && ropts.SecretKey != "" {
		opts = append(opts, producer.WithCredentials(primitive.Credentials{
			AccessKey: ropts.AccessKey,
			SecretKey: ropts.SecretKey,
		}))
	}

	client, err := rocketmq.NewProducer(opts...)

	if err != nil {
		return nil, err
	}

	r := &RocketMq{
		endPoints: endPoints,
		producer:  client,
	}

	err = r.producer.Start()
	if err != nil {
		return nil, err
	}
	return r, nil
}

/**
 * @description: 创建rocketmq消费者
 * @param {[]string} endPoints nameserver
 * @param {string} groupName
 * @return {*}
 */
func NewRocketMqConsumer(endPoints []string, groupName string, options ...RocketOption) (c Consumer, err error) {
	opts := []consumer.Option{
		consumer.WithNsResolver(primitive.NewPassthroughResolver(endPoints)),
		consumer.WithGroupName(groupName),
		consumer.WithConsumerModel(consumer.Clustering)}

	ropts := &rocketOptions{}
	for _, v := range options {
		v(ropts)
	}

	if ropts.AccessKey != "" && ropts.SecretKey != "" {
		opts = append(opts, consumer.WithCredentials(primitive.Credentials{
			AccessKey: ropts.AccessKey,
			SecretKey: ropts.SecretKey,
		}))
	}

	client, err := rocketmq.NewPushConsumer(opts...)

	if err != nil {
		return nil, err
	}

	r := &RocketMq{
		endPoints: endPoints,
		consumer:  client,
	}

	return r, nil
}
